# cPanel Deployment Guide for Peerpurse

This guide will help you deploy your Next.js application to cPanel with PostgreSQL database.

## Prerequisites

1. cPanel access with Node.js support enabled
2. PostgreSQL database access (create via cPanel)
3. SSH access (recommended) or File Manager access
4. Domain name or subdomain configured

---

## Step 1: Create PostgreSQL Database in cPanel

1. **Login to cPanel**
2. **Go to "PostgreSQL Databases"** (or search for it)
3. **Create a new database:**
   - Click "Create Database"
   - Name it: `peerpurse_db` (or your preferred name)
   - Note the full database name (usually `cpaneluser_peerpurse_db`)

4. **Create a database user:**
   - Go to "PostgreSQL Users"
   - Create a new user with a strong password
   - Note the username (usually `cpaneluser_dbuser`)

5. **Add user to database:**
   - Go to "Add User To Database"
   - Select your user and database
   - Grant ALL PRIVILEGES
   - Click "Make Changes"

6. **Note your connection details:**
   ```
   Host: localhost (or provided by your host)
   Database: cpaneluser_peerpurse_db
   User: cpaneluser_dbuser
   Password: [your password]
   Port: 5432 (usually)
   ```

---

## Step 2: Prepare Your Application for Deployment

### 2.1 Update next.config.ts for Production

Your `next.config.ts` should be configured for production. Update it if needed:

```typescript
import type { NextConfig } from "next";

const nextConfig: NextConfig = {
  output: 'standalone', // For easier deployment
};

export default nextConfig;
```

### 2.2 Build Your Application Locally

```bash
cd web
npm install
npm run build
```

This creates a `.next` folder with the production build.

---

## Step 3: Upload Files to cPanel

### Option A: Using File Manager

1. **Login to cPanel → File Manager**
2. **Navigate to your domain folder** (usually `public_html` or `subdomain_name`)
3. **Upload all files** from your `web/` folder:
   - Upload all files keeping the folder structure
   - Make sure to include:
     - `.next/` folder (production build)
     - `node_modules/` (or install on server)
     - `package.json`
     - `prisma/` folder
     - `app/` folder
     - `components/` folder
     - `lib/` folder
     - All config files

### Option B: Using FTP/SFTP (Recommended)

1. **Use an FTP client** (FileZilla, WinSCP, etc.)
2. **Connect to your server**
3. **Upload the entire `web/` folder** to your domain directory

---

## Step 4: Install Dependencies on Server

### Via SSH (Recommended):

```bash
cd /home/cpaneluser/public_html  # or your domain path
npm install --production
npx prisma generate
```

### Via cPanel Terminal:

1. Go to cPanel → Terminal
2. Navigate to your app directory
3. Run the commands above

---

## Step 5: Set Up Environment Variables

### Create `.env` file in your app directory:

```env
# Database
DATABASE_URL="postgresql://cpaneluser_dbuser:your_password@localhost:5432/cpaneluser_peerpurse_db"

# Admin Credentials
ADMIN_EMAIL=admin@yourdomain.com
ADMIN_PASSWORD=your_secure_password_here

# Session Secret (generate a random 32+ character string)
NEXTAUTH_SECRET=your-long-random-secret-string-at-least-32-characters

# Node Environment
NODE_ENV=production
```

**Important:** 
- Replace all placeholder values with your actual credentials
- Keep `.env` file secure and never commit it to git
- Use strong passwords

---

## Step 6: Run Database Migrations

Via SSH or cPanel Terminal:

```bash
cd /home/cpaneluser/public_html  # your app path
npx prisma migrate deploy
```

This will create all database tables.

---

## Step 7: Set Up Node.js App in cPanel

1. **Go to cPanel → "Node.js Selector"** (or "Setup Node.js App")

2. **Create a new application:**
   - **Node.js Version:** Select the latest LTS (18.x or 20.x)
   - **Application Root:** `/home/cpaneluser/public_html` (or your domain path)
   - **Application URL:** Choose your domain/subdomain
   - **Application Startup File:** `server.js` (we'll create this)
   - **Application Mode:** Production

3. **Create `server.js` in your app root:**

```javascript
const { createServer } = require('http');
const { parse } = require('url');
const next = require('next');

const dev = process.env.NODE_ENV !== 'production';
const hostname = 'localhost';
const port = process.env.PORT || 3000;

const app = next({ dev, hostname, port });
const handle = app.getRequestHandler();

app.prepare().then(() => {
  createServer(async (req, res) => {
    try {
      const parsedUrl = parse(req.url, true);
      await handle(req, res, parsedUrl);
    } catch (err) {
      console.error('Error occurred handling', req.url, err);
      res.statusCode = 500;
      res.end('internal server error');
    }
  }).listen(port, (err) => {
    if (err) throw err;
    console.log(`> Ready on http://${hostname}:${port}`);
  });
});
```

4. **Update package.json scripts** (if needed):

```json
{
  "scripts": {
    "start": "node server.js",
    "build": "next build",
    "postinstall": "prisma generate"
  }
}
```

5. **Click "Run" or "Restart"** in Node.js Selector

---

## Step 8: Configure Domain/DNS

1. **Point your domain** to the server IP
2. **Set up subdomain** if using one:
   - Go to cPanel → Subdomains
   - Create subdomain pointing to your app directory
   - Update DNS records if needed

---

## Step 9: Test Your Deployment

1. **Visit your domain** in a browser
2. **Test the waitlist form:**
   - Fill out and submit the form
   - Check if data saves to database

3. **Test admin login:**
   - Go to `/admin/login`
   - Login with your admin credentials
   - Verify dashboard shows entries

---

## Troubleshooting

### Issue: "Cannot find module" errors
**Solution:** Make sure `node_modules/` is uploaded or run `npm install` on server

### Issue: Database connection errors
**Solution:** 
- Verify DATABASE_URL is correct
- Check database user has proper permissions
- Verify PostgreSQL is running

### Issue: 500 errors
**Solution:**
- Check server logs in cPanel
- Verify environment variables are set
- Ensure Prisma migrations ran successfully

### Issue: Static files not loading
**Solution:**
- Verify `.next/` folder is uploaded
- Check file permissions (should be 755 for folders, 644 for files)

### Issue: Port conflicts
**Solution:**
- Check Node.js app port in cPanel
- Verify no other apps use the same port

---

## Security Checklist

- [ ] Use strong passwords for database and admin
- [ ] Set NODE_ENV=production
- [ ] Keep `.env` file secure (600 permissions)
- [ ] Enable HTTPS/SSL certificate
- [ ] Regularly update dependencies
- [ ] Set up firewall rules if possible
- [ ] Regular database backups

---

## Alternative: Using PM2 (If Available)

If you have SSH access and can install PM2:

```bash
npm install -g pm2
pm2 start server.js --name peerpurse
pm2 save
pm2 startup
```

---

## Need Help?

- Check cPanel logs: `cPanel → Metrics → Errors`
- Check Node.js logs in Node.js Selector
- Verify database connection in cPanel
- Review server error logs

---

## Quick Reference Commands

```bash
# Build app
npm run build

# Generate Prisma client
npx prisma generate

# Run migrations
npx prisma migrate deploy

# Start production server
npm start

# Check Prisma connection
npx prisma db pull
```

