import { NextRequest, NextResponse } from "next/server";
import { prisma } from "@/lib/prisma";
import { waitlistSchema } from "@/lib/validation";
import { getAdminSession } from "@/lib/session";

export async function POST(req: NextRequest) {
    try {
        const data = await req.json();
        const parsed = waitlistSchema.safeParse(data);
        if (!parsed.success) {
            return NextResponse.json(
                { error: "Invalid input", details: parsed.error.flatten() },
                { status: 400 }
            );
        }

        const { name, email, phone } = parsed.data;

        const created = await prisma.waitingList.create({
            data: { name, email: email.toLowerCase(), phone },
        });

        return NextResponse.json({ id: created.id }, { status: 201 });
    } catch (err: unknown) {
        if (err && typeof err === "object" && "code" in err && err.code === "P2002") {
            return NextResponse.json(
                { error: "Email already registered" },
                { status: 409 }
            );
        }
        return NextResponse.json({ error: "Server error" }, { status: 500 });
    }
}

export async function GET() {
    const session = await getAdminSession();
    if (!session.isLoggedIn) {
        return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }

    const entries = await prisma.waitingList.findMany({
        orderBy: { createdAt: "desc" },
    });
    return NextResponse.json(entries);
}


