"use client";
import { useEffect, useMemo, useState } from "react";
import { ColumnDef, flexRender, getCoreRowModel, getSortedRowModel, useReactTable } from "@tanstack/react-table";

type Entry = {
    id: number;
    name: string;
    email: string;
    phone: string;
    createdAt: string;
};

export default function AdminTable() {
    const [data, setData] = useState<Entry[]>([]);
    const [query, setQuery] = useState("");
    const [page, setPage] = useState(1);
    const pageSize = 10;

    useEffect(() => {
        fetch("/api/waitlist")
            .then((r) => (r.ok ? r.json() : Promise.reject()))
            .then((d: Entry[]) => setData(d))
            .catch(() => setData([]));
    }, []);

    const filtered = useMemo(() => {
        const q = query.trim().toLowerCase();
        if (!q) return data;
        return data.filter((e) => e.name.toLowerCase().includes(q) || e.email.toLowerCase().includes(q));
    }, [data, query]);

    const paged = useMemo(() => {
        const start = (page - 1) * pageSize;
        return filtered.slice(start, start + pageSize);
    }, [filtered, page]);

    useEffect(() => {
        setPage(1);
    }, [query]);

    const columns = useMemo<ColumnDef<Entry>[]>(
        () => [
            { accessorKey: "name", header: () => "Name" },
            { accessorKey: "email", header: () => "Email" },
            { accessorKey: "phone", header: () => "Phone" },
            {
                accessorKey: "createdAt",
                header: () => "Date Joined",
                cell: ({ getValue }) => new Date(getValue<string>()).toLocaleString(),
            },
        ],
        []
    );

    const table = useReactTable({ data: paged, columns, getCoreRowModel: getCoreRowModel(), getSortedRowModel: getSortedRowModel() });

    function exportCSV() {
        const header = ["Name", "Email", "Phone", "Date Joined"];
        const rows = filtered.map((e) => [e.name, e.email, e.phone, new Date(e.createdAt).toISOString()]);
        const csv = [header, ...rows].map((r) => r.map((c) => `"${String(c).replaceAll('"', '""')}"`).join(",")).join("\n");
        const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
        const url = URL.createObjectURL(blob);
        const a = document.createElement("a");
        a.href = url;
        a.download = `waiting-list-${new Date().toISOString()}.csv`;
        a.click();
        URL.revokeObjectURL(url);
    }

    return (
        <div className="space-y-4">
            <div className="flex items-center gap-2">
                <input
                    placeholder="Search by name or email..."
                    value={query}
                    onChange={(e) => setQuery(e.target.value)}
                    className="w-full sm:w-80 rounded-md border border-gray-300 px-3 py-2 focus:outline-none focus:ring-2 focus:ring-[#6B21A8]"
                />
                <button onClick={exportCSV} className="rounded-md bg-[#6B21A8] text-white px-4 py-2">Export CSV</button>
            </div>

            <div className="overflow-x-auto border rounded-lg">
                <table className="min-w-full text-sm">
                    <thead className="bg-gray-50">
                        {table.getHeaderGroups().map((hg) => (
                            <tr key={hg.id}>
                                {hg.headers.map((h) => (
                                    <th key={h.id} className="px-4 py-2 text-left font-medium text-gray-600">
                                        {h.isPlaceholder ? null : (
                                            <div>{flexRender(h.column.columnDef.header, h.getContext())}</div>
                                        )}
                                    </th>
                                ))}
                            </tr>
                        ))}
                    </thead>
                    <tbody>
                        {table.getRowModel().rows.map((row) => (
                            <tr key={row.id} className="odd:bg-white even:bg-gray-50">
                                {row.getVisibleCells().map((cell) => (
                                    <td key={cell.id} className="px-4 py-2 text-gray-800">
                                        {flexRender(cell.column.columnDef.cell, cell.getContext())}
                                    </td>
                                ))}
                            </tr>
                        ))}
                        {paged.length === 0 && (
                            <tr>
                                <td className="px-4 py-6 text-center text-gray-500" colSpan={columns.length}>No entries</td>
                            </tr>
                        )}
                    </tbody>
                </table>
            </div>

            <div className="flex items-center justify-between">
                <div className="text-sm text-gray-600">
                    Page {page} of {Math.max(1, Math.ceil(filtered.length / pageSize))}
                </div>
                <div className="space-x-2">
                    <button
                        onClick={() => setPage((p) => Math.max(1, p - 1))}
                        className="rounded-md border px-3 py-1.5 disabled:opacity-50"
                        disabled={page === 1}
                    >
                        Previous
                    </button>
                    <button
                        onClick={() => setPage((p) => (p * pageSize < filtered.length ? p + 1 : p))}
                        className="rounded-md border px-3 py-1.5 disabled:opacity-50"
                        disabled={page * pageSize >= filtered.length}
                    >
                        Next
                    </button>
                </div>
            </div>
        </div>
    );
}


