"use client";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { waitlistSchema, type WaitlistInput } from "@/lib/validation";
import { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import toast, { Toaster } from "react-hot-toast";

export default function WaitlistForm() {
    const [success, setSuccess] = useState(false);
    const {
        register,
        handleSubmit,
        formState: { errors, isSubmitting },
        reset,
    } = useForm<WaitlistInput>({ resolver: zodResolver(waitlistSchema) });

    async function onSubmit(values: WaitlistInput) {
        const res = await fetch("/api/waitlist", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(values),
        });
        if (res.ok) {
            toast.success("Thanks for joining our waiting list! We’ll notify you once we launch.");
            setSuccess(true);
            reset();
        } else {
            const data = await res.json().catch(() => ({}));
            toast.error(data?.error || "Something went wrong");
        }
    }

    return (
        <section id="join" className="relative w-full py-20">
            <div className="absolute inset-x-0 top-0 -z-10 h-60 bg-gradient-to-b from-purple-50 to-transparent" />
            <div className="max-w-2xl mx-auto px-6">
                <div className="rounded-2xl border border-white/60 bg-white/70 backdrop-blur-xl p-6 shadow-md">
                    <h2 className="text-2xl font-semibold text-gray-900">Join the waiting list</h2>
                    <p className="text-gray-600 mt-2">Be first to know when we launch.</p>
                    <form onSubmit={handleSubmit(onSubmit)} className="mt-6 grid gap-4">
                        <div>
                            <label className="block text-sm font-medium text-gray-700">Full Name</label>
                            <input
                                type="text"
                                {...register("name")}
                                className="mt-1 w-full rounded-lg border border-gray-300/80 bg-white/80 px-3 py-2 text-black placeholder:text-gray-400 focus:outline-none focus:ring-2 focus:ring-[#6B21A8]"
                                placeholder="Jane Doe"
                            />
                            {errors.name && (
                                <p className="text-sm text-red-600 mt-1">{errors.name.message}</p>
                            )}
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700">Email Address</label>
                            <input
                                type="email"
                                {...register("email")}
                                className="mt-1 w-full rounded-lg border border-gray-300/80 bg-white/80 px-3 py-2 text-black placeholder:text-gray-400 focus:outline-none focus:ring-2 focus:ring-[#6B21A8]"
                                placeholder="jane@example.com"
                            />
                            {errors.email && (
                                <p className="text-sm text-red-600 mt-1">{errors.email.message}</p>
                            )}
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700">Phone Number</label>
                            <input
                                type="tel"
                                {...register("phone")}
                                className="mt-1 w-full rounded-lg border border-gray-300/80 bg-white/80 px-3 py-2 text-black placeholder:text-gray-400 focus:outline-none focus:ring-2 focus:ring-[#6B21A8]"
                                placeholder="+1 555 123 4567"
                            />
                            {errors.phone && (
                                <p className="text-sm text-red-600 mt-1">{errors.phone.message}</p>
                            )}
                        </div>
                        <button
                            type="submit"
                            disabled={isSubmitting}
                            className="mt-2 w-full rounded-xl bg-[#6B21A8] text-white py-2.5 font-medium shadow-[0_10px_30px_-10px_rgba(107,33,168,0.7)] hover:opacity-95 disabled:opacity-60 disabled:cursor-not-allowed"
                        >
                            {isSubmitting ? "Submitting..." : "Join Waiting List"}
                        </button>
                    </form>
                    <Toaster position="top-right" />
                    <AnimatePresence>
                        {success && (
                            <motion.div
                                initial={{ opacity: 0, y: 10 }}
                                animate={{ opacity: 1, y: 0 }}
                                exit={{ opacity: 0, y: 10 }}
                                className="mt-6 rounded-lg border border-purple-100 bg-purple-50 p-4 text-purple-900"
                                onAnimationComplete={() => setSuccess(false)}
                            >
                                ✅ Thanks for joining our waiting list! We’ll notify you once we launch.
                            </motion.div>
                        )}
                    </AnimatePresence>
                </div>
            </div>
        </section>
    );
}


