"use client";
import { useEffect } from "react";
import { motion, AnimatePresence } from "framer-motion";
import WaitlistForm from "@/components/WaitlistForm";

type Props = { open: boolean; onClose: () => void };

export default function WaitlistModal({ open, onClose }: Props) {
    useEffect(() => {
        function onKey(e: KeyboardEvent) {
            if (e.key === "Escape") onClose();
        }
        if (open) document.addEventListener("keydown", onKey);
        return () => document.removeEventListener("keydown", onKey);
    }, [open, onClose]);

    return (
        <AnimatePresence>
            {open && (
                <motion.div
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    exit={{ opacity: 0 }}
                    className="fixed inset-0 z-50 pointer-events-none"
                >
                    {/* Overlay */}
                    <div
                        onClick={onClose}
                        className="absolute inset-0 bg-black/30 backdrop-blur-sm z-0 cursor-pointer pointer-events-auto"
                    />
                    {/* Dialog */}
                    <div className="absolute inset-0 z-10 flex items-center justify-center px-4 py-8 pointer-events-auto">
                        <motion.div
                            initial={{ opacity: 0, scale: 0.96, y: 8 }}
                            animate={{ opacity: 1, scale: 1, y: 0 }}
                            exit={{ opacity: 0, scale: 0.96, y: 8 }}
                            transition={{ duration: 0.18 }}
                            role="dialog"
                            aria-modal="true"
                            onClick={(e) => e.stopPropagation()}
                            className="relative w-full max-w-lg rounded-2xl border border-white/60 bg-white/90 backdrop-blur-xl shadow-2xl"
                        >
                            <button
                                type="button"
                                aria-label="Close"
                                onClick={(e) => { e.stopPropagation(); e.preventDefault(); onClose(); }}
                                onMouseDown={(e) => { e.stopPropagation(); }}
                                onTouchStart={(e) => { e.stopPropagation(); }}
                                className="absolute right-4 top-4 z-20 inline-flex h-12 w-12 items-center justify-center rounded-full border-2 border-gray-300 bg-white/90 backdrop-blur-sm text-gray-700 hover:border-[#6B21A8] hover:bg-purple-50 hover:text-[#6B21A8] focus:outline-none focus:ring-2 focus:ring-[#6B21A8]/40 cursor-pointer shadow-md transition-all duration-200 active:scale-95"
                                style={{ touchAction: 'manipulation' }}
                            >
                                {/* Use a scalable icon for a larger, reliable hit area */}
                                <svg
                                    xmlns="http://www.w3.org/2000/svg"
                                    viewBox="0 0 24 24"
                                    fill="none"
                                    stroke="currentColor"
                                    strokeWidth="2.5"
                                    className="h-6 w-6 pointer-events-none select-none"
                                    aria-hidden="true"
                                >
                                    <path strokeLinecap="round" strokeLinejoin="round" d="M6 18L18 6M6 6l12 12" />
                                </svg>
                            </button>
                            <div className="p-5">
                                <WaitlistForm />
                            </div>
                        </motion.div>
                    </div>
                </motion.div>
            )}
        </AnimatePresence>
    );
}


